<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   (C) 2023 JoomShaper. <https://www.joomshaper.com>
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

namespace JoomShaper\Component\EasyStore\Site\Controller;

use Joomla\CMS\Factory;
use Joomla\Input\Input;
use Joomla\Input\Cookie;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Event\AbstractEvent;
use Joomla\CMS\Layout\LayoutHelper;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Application\CMSApplication;
use Joomla\CMS\MVC\Controller\BaseController;
use Joomla\CMS\MVC\Factory\MVCFactoryInterface;
use JoomShaper\Component\EasyStore\Site\Traits\Api;
use JoomShaper\Component\EasyStore\Site\Traits\Cart;
use JoomShaper\Component\EasyStore\Site\Model\CartModel;
use JoomShaper\Component\EasyStore\Site\Helper\EasyStoreHelper as SiteEasyStoreHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Default Controller of EasyStore component
 *
 * @since  1.0.0
 */
class CartController extends BaseController
{
    use Api;
    use Cart;

    /**
     * The cart cookie constant
     *
     * @var string
     */
    public const TOKEN_COOKIE_NAME = 'com_easystore_cart';

    protected $cookie = null;

    public function __construct($config = [], MVCFactoryInterface $factory = null, ?CMSApplication $app = null, ?Input $input = null)
    {
        $this->cookie = new Cookie();

        parent::__construct($config, $factory, $app, $input);
    }

    public function getCartData()
    {
        $model  = new CartModel();
        $item   = $model->getItem();

        $this->sendResponse($item);
    }

    public function getCartDrawerContent()
    {
        $model       = new CartModel();
        $item        = $model->getItem();
        $output      = LayoutHelper::render('cart.mini', ['items' => $item ?? []]);
        $token       = $this->getToken();
        $checkoutUrl = Route::_('index.php?option=com_easystore&view=checkout&cart_token=' . $token, false);

        $this->sendResponse(['content' => $output, 'checkout_url' => $checkoutUrl]);
    }

    /**
     * Payment function.
     *
     * This function handles the payment process based on the selected payment type.
     * It initiates the payment process, imports the corresponding payment plugin,
     * and dispatches an event to trigger the payment functionality.
     *
     * @return void
     * @since 1.0.0
     */
    public function payment()
    {
        $paymentType = $this->input->post->get('paymentType', '', 'STRING');

        if (!PluginHelper::isEnabled('easystore', $paymentType)) {
            $message = Text::sprintf('COM_EASYSTORE_CART_PLUGIN_IS_DISABLE', ucfirst($paymentType), 'error');
            $output  = SiteEasyStoreHelper::showMessage('error', $message);
            echo $output;
        }

        PluginHelper::importPlugin('easystore', $paymentType);

        $cparams  = ComponentHelper::getParams('com_easystore');
        $currency = $cparams->get('currency', 'USD:$');


        $paymentInfo                     = new \stdClass();
        $paymentInfo->currency           = $currency;
        $paymentInfo->productName        = 'New Product';
        $paymentInfo->productDescription = "Description";
        $paymentInfo->amount             = '300';

        $event = AbstractEvent::create(
            'onPayment',
            [
                'subject' => $paymentInfo,
            ]
        );

        try {
            Factory::getApplication()->getDispatcher()->dispatch($event->getName(), $event);
        } catch (\Throwable $th) {
            Factory::getApplication()->enqueueMessage($th->getMessage());
        }
    }
}
