<?php

/**
 * @package     EasyStore.Site
 * @subpackage  com_easystore
 *
 * @copyright   Copyright (C) 2023 JoomShaper <https://www.joomshaper.com>. All rights reserved.
 * @license     GNU General Public License version 3; see LICENSE
 */

namespace JoomShaper\Component\EasyStore\Site\Model;

use Joomla\CMS\Factory;
use Joomla\CMS\MVC\Model\ListModel;
use JoomShaper\Component\EasyStore\Administrator\Model\ProductModel;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreHelper;
use JoomShaper\Component\EasyStore\Administrator\Helper\EasyStoreDatabaseOrm;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects


class OrdersModel extends ListModel
{
    /**
     * Model context string.
     *
     * @var    string
     * @since  1.0.0
     */
    public $_context = 'com_easystore.orders';

    /**
     * Method to build an SQL query to load the list data.
     *
     * @return  DatabaseQuery  An SQL query
     *
     * @since   1.0.0
     */
    protected function getListQuery()
    {
        $app         = Factory::getApplication();
        $loginUserId = $app->getIdentity()->id;

        $easystoreUser = EasyStoreDatabaseOrm::get('#__easystore_users', 'user_id', $loginUserId)->loadObject();

        // Create a new query object.
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);

        $query->select(['*'])
            ->from($db->quoteName('#__easystore_orders', 'o'))
            ->where($db->quoteName('o.customer_id') . ' = ' . $db->quote($easystoreUser->id ?? ''))
            ->where($db->quoteName('o.published') . ' = 1')
            ->order($db->quoteName('o.id') . 'DESC');

        return $query;
    }

    /**
     * Method to get an array of data items.
     *
     * @return  mixed  An array of data items on success, false on failure.
     *
     * @since   1.0.0
     */
    public function getItems()
    {
        $items = parent::getItems();

        foreach ($items as $item) {
            $db    = $this->getDatabase();
            $query = $db->getQuery(true);
            // @todo we will update this query for admin discount
            $subquery = $db->getQuery(true);
            $subquery->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity']))
            ->select('SUM(' . $db->quoteName('price') . ' * ' . $db->quoteName('quantity') . ') AS sub_total')
            ->from($db->quoteName('#__easystore_order_product_map'))
            ->where($db->quoteName('order_id') . ' = ' . $item->id)
            ->group([$db->quoteName('order_id'), $db->quoteName('discount_type'), $db->quoteName('discount_value'), $db->quoteName('price'), $db->quoteName('quantity')]);

            // @todo we will update this query for admin discount
            $query->select($db->quoteName(['order_id', 'discount_type', 'discount_value', 'price', 'quantity', 'sub_total']))
            ->select('
                CASE WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                    CASE 
                        WHEN ' . $db->quoteName('discount_type') . ' = ' . $db->quote('percent') . ' THEN 
                        ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                        ELSE ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                    END 
                ELSE
                    0.00
                END AS discounted_price,

                CASE
                    WHEN (
                        CASE
                            WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                CASE
                                    WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                       ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                    ELSE
                                       ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                END
                            ELSE
                                0.00
                        END
                    ) = 0.00 THEN 0.00
                    ELSE
                    ' . $db->quoteName('price') . ' - (
                            CASE
                                WHEN ' . $db->quoteName('discount_value') . ' > 0 THEN
                                    CASE
                                        WHEN ' . $db->quoteName('discount_type') . ' = "percent" THEN
                                           ' . $db->quoteName('price') . ' - (' . $db->quoteName('price') . ' * (' . $db->quoteName('discount_value') . ') / 100)
                                        ELSE
                                           ' . $db->quoteName('price') . ' - ' . $db->quoteName('discount_value') . '
                                    END
                                ELSE
                                    0.00
                            END
                        )
                END  AS special_discounted_amount
                ')
            ->from('(' . $subquery . ') AS subquery');


            // Execute the query
            $db->setQuery($query);

            $orderedProducts                 = $db->loadObjectList();
            $item->special_discounted_amount = 0.00;
            $item->shipping_cost             = 0.00;
            $item->coupon_discount           = 0.00;
            $item->discounted_sub_total      = 0.00;
            $item->sub_total                 = 0.00;

            if (!empty($orderedProducts)) {
                foreach ($orderedProducts as &$product) {
                    $item->sub_total += floatval($product->sub_total);

                    // @todo  we will add this in future
                    if (floatval($product->discounted_price) > 0) {
                        $item->discounted_sub_total += floatval($product->discounted_price);
                    }

                    if (!is_null(floatval($product->special_discounted_amount))) {
                        $item->special_discounted_amount += floatval($product->special_discounted_amount);
                    }
                }

                unset($product);
            }

            if (!is_null($item->coupon_amount) && floatval($item->coupon_amount) > 0) {
                $item->coupon_discount = EasyStoreHelper::calculateDiscountValue($item->coupon_type, $item->coupon_amount, $item->sub_total);
            }
            // @todo  we will add this in future
            if (!empty($item->special_discounted_amount)) {
                $totalPrice = floatval($item->sub_total - $item->special_discounted_amount);
            }

            if (floatval($item->discount_value) > 0) {
                $totalPrice = EasyStoreHelper::calculateDiscountedPrice($item->discount_type, $item->discount_value, $totalPrice);
            }

            if (!empty($item->shipping) && is_string($item->shipping)) {
                // Convert the shipping string to an object
                $shippingData = json_decode($item->shipping);

                if (isset($shippingData->offerFreeShipping) && $shippingData->offerFreeShipping) {
                    // Check if there's an offer on a specific amount
                    $offerOnAmount = (float) ($shippingData->offerOnAmount ?? null);

                    if ($item->sub_total > $offerOnAmount) {
                        // Apply free shipping if the subtotal is above the offer amount
                        $shippingData->rate = 0;
                    }
                }

                if (isset($shippingData->rate)) {
                    // Format the shipping rate with currency
                    $shippingData->rate_with_currency = EasyStoreHelper::formatCurrency($shippingData->rate);
                }

                // Update the shipping cost with the calculated rate or default to 0
                $item->shipping_cost = $shippingData->rate ?? 0;
            }

            $totalPrice = $item->sub_total + $item->shipping_cost + $item->sale_tax - $item->coupon_discount;

            $item->totalPrice = EasyStoreHelper::formatCurrency($totalPrice);
            $item->published  = EasyStoreHelper::getOrderStatusName($item->order_status);
        }

        $items = $this->getProductsOfItems($items);

        return $items;
    }

    /**
     * Function to get Products info of every Item
     *
     * @param Object $items
     * @return object
     */
    private function getProductsOfItems($items)
    {
        $orm = new EasyStoreDatabaseOrm();

        foreach ($items as &$item) {
            $products = $orm->hasMany($item->id, '#__easystore_order_product_map', 'order_id')
                ->loadObjectList();
            $item->products = $products;

            foreach ($products as &$product) {
                $productData    = ProductModel::getProductDataWithImage($product->product_id, $item->id);
                $product->image = $productData->image;
            }

            unset($product);
        }

        unset($item);

        return $items;
    }
}
